<?php
header('Content-Type: application/json');
require '../db_connect.php';

// --- ধাপ ১: API Key দিয়ে ব্যবহারকারী এবং তার টাইমজোন খুঁজে বের করা ---
$api_key = $_SERVER['HTTP_X_API_KEY'] ?? '';
if (empty($api_key)) { 
    http_response_code(401); 
    echo json_encode(['success' => false, 'message' => 'API Key is missing.']); 
    exit(); 
}

// SQL কোয়েরি পরিবর্তন করে settings টেবিলের সাথে brand_settings টেবিল JOIN করা হয়েছে
$sql_user = "SELECT s.user_id, bs.timezone 
             FROM settings s
             LEFT JOIN brand_settings bs ON s.user_id = bs.user_id
             WHERE s.api_key = ?";
$stmt_user = $pdo->prepare($sql_user);
$stmt_user->execute([$api_key]);
$setting = $stmt_user->fetch(PDO::FETCH_ASSOC);

if (!$setting) { 
    http_response_code(403); 
    echo json_encode(['success' => false, 'message' => 'Invalid API Key.']); 
    exit(); 
}
$user_id = $setting['user_id'];
// ব্যবহারকারীর সেট করা টাইমজোন নেওয়া হচ্ছে, সেট করা না থাকলে ডিফল্ট হিসেবে 'Asia/Dhaka' ব্যবহার করা হবে
$user_timezone = $setting['timezone'] ?? 'Asia/Dhaka';

// --- Configuration & Functions ---
define('BDT_TO_USD_RATE', 0.0085);

/**
 * নির্দিষ্ট সময়কালের জন্য পেমেন্টের ডেটা তৈরি করে।
 * @param PDO $pdo PDO ডাটাবেস অবজেক্ট
 * @param int $user_id ব্যবহারকারীর আইডি
 * @param string $start_date_utc UTC ফরম্যাটে শুরুর তারিখ
 * @param string $end_date_utc UTC ফরম্যাটে শেষের তারিখ
 * @return array রিপোর্ট ডেটা
 */
function get_report_data($pdo, $user_id, $start_date_utc, $end_date_utc) {
    // ★★★ মূল পরিবর্তন: এখন এখানে আর CONVERT_TZ নেই ★★★
    // সরাসরি UTC সময় অনুযায়ী ডেটা খোঁজা হচ্ছে
    $sql_completed = "SELECT 
                        COALESCE(SUM(amount), 0) as total_amount, 
                        COUNT(id) as payment_count 
                      FROM transactions 
                      WHERE user_id = ? AND status = 'completed' AND created_at BETWEEN ? AND ?";
    
    $stmt_completed = $pdo->prepare($sql_completed);
    $stmt_completed->execute([$user_id, $start_date_utc, $end_date_utc]);
    $completed_data = $stmt_completed->fetch(PDO::FETCH_ASSOC);

    $sql_initiated = "SELECT COUNT(id) as total_count 
                      FROM transactions 
                      WHERE user_id = ? AND created_at BETWEEN ? AND ?";
    
    $stmt_initiated = $pdo->prepare($sql_initiated);
    $stmt_initiated->execute([$user_id, $start_date_utc, $end_date_utc]);
    $initiated_count = $stmt_initiated->fetchColumn() ?? 0;

    $success_rate = 0;
    if ($initiated_count > 0) {
        $success_rate = ($completed_data['payment_count'] / $initiated_count) * 100;
    }

    return [
        'total_amount_bdt' => (float)$completed_data['total_amount'],
        'total_amount_usd' => (float)$completed_data['total_amount'] * BDT_TO_USD_RATE,
        'payment_count' => (int)$completed_data['payment_count'],
        'success_rate' => number_format($success_rate, 1) . '% success'
    ];
}

// --- ধাপ ৩: রিপোর্ট তৈরি এবং পাঠানো ---
$reports = [];
$icon_colors = ["#3498db", "#e74c3c", "#2ecc71", "#f1c40f", "#9b59b6", "#1abc9c", "#e67e22", "#34495e"];
$color_index = 0;

// ★★★ মূল পরিবর্তন: এখন PHP-তেই সময় রূপান্তর করা হচ্ছে ★★★
$user_tz_obj = new DateTimeZone($user_timezone);
$utc_tz_obj = new DateTimeZone('UTC');

$periods = [
    'Today' => [new DateTime('today 00:00:00', $user_tz_obj), new DateTime('today 23:59:59', $user_tz_obj)],
    'Yesterday' => [new DateTime('yesterday 00:00:00', $user_tz_obj), new DateTime('yesterday 23:59:59', $user_tz_obj)],
    'This Week' => [new DateTime('monday this week 00:00:00', $user_tz_obj), new DateTime('sunday this week 23:59:59', $user_tz_obj)],
    'Last Week' => [new DateTime('monday last week 00:00:00', $user_tz_obj), new DateTime('sunday last week 23:59:59', $user_tz_obj)],
    'This Month' => [new DateTime('first day of this month 00:00:00', $user_tz_obj), new DateTime('last day of this month 23:59:59', $user_tz_obj)],
    'Last Month' => [new DateTime('first day of last month 00:00:00', $user_tz_obj), new DateTime('last day of last month 23:59:59', $user_tz_obj)],
    'This Year' => [new DateTime('first day of January this year 00:00:00', $user_tz_obj), new DateTime('last day of December this year 23:59:59', $user_tz_obj)],
    'Last Year' => [new DateTime('first day of January last year 00:00:00', $user_tz_obj), new DateTime('last day of December last year 23:59:59', $user_tz_obj)]
];

foreach ($periods as $title => $datetimes) {
    // লোকাল সময়কে UTC-তে রূপান্তর করা হচ্ছে
    $start_utc = $datetimes[0]->setTimezone($utc_tz_obj)->format('Y-m-d H:i:s');
    $end_utc = $datetimes[1]->setTimezone($utc_tz_obj)->format('Y-m-d H:i:s');
    
    // get_report_data ফাংশনে এখন UTC সময় পাঠানো হচ্ছে
    $data = get_report_data($pdo, $user_id, $start_utc, $end_utc);
    $data['title'] = $title;
    $data['icon_color'] = $icon_colors[$color_index % count($icon_colors)];
    $reports[] = $data;
    $color_index++;
}

// চূড়ান্ত ফলাফল JSON ফরম্যাটে পাঠানো
echo json_encode($reports);
?>