<?php
require 'auth_check.php';
require 'db_connect.php';

// ডাইনামিক URL এবং API Key আনা হচ্ছে
$user_id = $_SESSION['user_id'];
$stmt = $pdo->prepare("SELECT api_key, base_url FROM settings WHERE user_id = ?");
$stmt->execute([$user_id]);
$setting = $stmt->fetch(PDO::FETCH_ASSOC);

$api_key = $setting['api_key'] ?? 'YOUR_SECRET_API_KEY';
$base_url = $setting['base_url'] ?? 'https://your-domain.com/api'; // ফলব্যাক URL

$endpoint_initiate = "{$base_url}/initiate_payment.php";
$endpoint_verify = "{$base_url}/verify_payment.php";

include 'templates/header.php';
include 'templates/sidebar.php';
?>
<div class="page-container">
    <div class="page-header">
        <h1 class="page-title">Developer API Documentation (v2.0)</h1>
    </div>

    <!-- Introduction -->
    <div class="content-card">
        <h3>Introduction & Authentication</h3>
        <p>This documentation provides a comprehensive guide to integrating our payment gateway. All API requests must be authenticated by including your API key in the request header.</p>
        <pre><code>Authorization: Bearer <?php echo htmlspecialchars($api_key); ?></code></pre>
        <p><em>(For legacy systems, you may also use <code>X-API-KEY: <?php echo htmlspecialchars($api_key); ?></code>)</em></p>
    </div>

    <!-- New Payment Flow -->
    <div class="content-card">
        <h3>Payment Flow Overview (Browser-Based Verification)</h3>
        <p>The integration process is now simpler and more reliable, relying on the user's browser to confirm the payment. The flow follows these steps:</p>
        <ol>
            <li><strong>Step 1: Initiate Payment.</strong> Your server sends a request to our <code>/initiate_payment.php</code> endpoint with the amount, customer details, and your success/cancel URLs.</li>
            <li><strong>Step 2: Redirect User.</strong> Our API responds with a unique <code>payment_url</code>. Your application must immediately redirect the user's browser to this URL to our secure checkout page.</li>
            <li><strong>Step 3: Customer Pays.</strong> The customer completes the payment on our secure checkout page.</li>
            <li><strong>Step 4: Return to Your Site with Proof.</strong> After a successful payment, our gateway redirects the user's browser back to the <code>success_url</code> you provided, but with an important addition: the unique <strong><code>transaction_id</code></strong> is appended to the URL as a query parameter.</li>
            <li><strong>Step 5: Verify on Your Server.</strong> Your script at the <code>success_url</code> must read this <code>transaction_id</code> from the URL. You should then update your database to mark the corresponding order as "Paid" and show a success message to the customer. This is the most crucial step.</li>
        </ol>
    </div>

    <!-- 1. Initiate Payment -->
    <div class="content-card">
        <h3>1. Initiate a Payment</h3>
        <p>To start a payment, send a POST request to the Payment Initiation Endpoint.</p>
        <h4>Endpoint</h4>
        <pre><code>POST <?php echo htmlspecialchars($endpoint_initiate); ?></code></pre>

        <h4>Request Body (JSON)</h4>
        <pre><code>{
    "amount": 150.50,
    "customer_name": "John Doe",
    "customer_email": "john.doe@example.com",
    "success_url": "https://yourwebsite.com/order/success.php",
    "cancel_url": "https://yourwebsite.com/cart.php",
    "meta_data": "{\"order_id\": \"YOUR_UNIQUE_ORDER_ID_123\"}"
}</code></pre>
        <h4>Parameters</h4>
        <ul>
            <li><code>amount</code> (float, required): The total amount to be paid.</li>
            <li><code>success_url</code> (string, required): The URL on your site where the customer will be redirected after a successful payment.</li>
            <li><code>cancel_url</code> (string, required): The URL where the customer is sent if they cancel.</li>
            <li><code>meta_data</code> (JSON string, required): A JSON string containing your internal order ID (<code>order_id</code>).</li>
        </ul>
        <h4>Success Response (JSON)</h4>
        <pre><code>{
    "status": "success",
    "payment_url": "https://securesoftpay.settings.top/dashboard_project/checkout.php?trx_id=a1b2c3..."
}
// Your server must now redirect the user's browser to the value of "payment_url".</code></pre>
    </div>

    <!-- 2. Handle the Return -->
    <div class="content-card">
        <h3>2. Handle the Customer's Return & Verify Payment</h3>
        <p>This is the most critical step. After a successful payment, the user will be redirected to your <code>success_url</code> with the payment details appended as query parameters.</p>

        <h4>Example Return URL:</h4>
        <p>The user's browser will be sent to a URL like this:</p>
        <pre><code>https://yourwebsite.com/order/success.php?success=true&transaction_id=CJH1D2Y4W1</code></pre>
        
        <h4>What Your `success.php` Script Must Do:</h4>
        <ol>
            <li>Check if a <code>transaction_id</code> exists in the URL.</li>
            <li>Retrieve your internal order ID from your session or database.</li>
            <li>Securely update your order's status to "Paid" or "Completed" in your database, storing the received <code>transaction_id</code> as proof of payment.</li>
            <li>Display a "Thank You" or order confirmation message to the customer.</li>
        </ol>
        
        <h4>Code Examples for Reading the `transaction_id`:</h4>
        
        <p><strong>Plain PHP:</strong></p>
        <pre><code>$transactionId = $_GET['transaction_id'] ?? null;
if ($transactionId) {
    // Update your database here...
}</code></pre>

        <p><strong>WordPress (in a custom page template):</strong></p>
        <pre><code>global $wp;
$transactionId = isset($_GET['transaction_id']) ? sanitize_text_field($_GET['transaction_id']) : null;
if ($transactionId) {
    $order_id = get_query_var('order_id'); // Example if using custom rewrite rules
    // update_post_meta($order_id, '_payment_status', 'completed');
    // update_post_meta($order_id, '_transaction_id', $transactionId);
}</code></pre>

        <p><strong>Laravel (in a Controller method):</strong></p>
        <pre><code>use Illuminate\Http\Request;

public function paymentSuccess(Request $request)
{
    $transactionId = $request->input('transaction_id');
    if ($transactionId) {
        // $order = Order::find($request->session()->get('order_id'));
        // $order->status = 'completed';
        // $order->transaction_id = $transactionId;
        // $order->save();
    }
}</code></pre>

        <p><strong>Node.js (with Express):</strong></p>
        <pre><code>app.get('/order/success', (req, res) => {
    const transactionId = req.query.transaction_id;
    if (transactionId) {
        // Find order in your database and update it
        // db.orders.updateOne({ _id: req.session.orderId }, { $set: { status: 'paid', transactionId: transactionId } });
    }
});</code></pre>
    </div>

     <!-- 3. Final Server-to-Server Verification (Optional) -->
    <div class="content-card">
        <h3>3. Final Server-to-Server Verification (Optional but Recommended)</h3>
        <p>For maximum security, after receiving the <code>transaction_id</code> on your <code>success_url</code>, you can make one final, secure, server-to-server API call to our verification endpoint to confirm the payment details before marking the order as paid.</p>
        <h4>Endpoint</h4>
        <pre><code>POST <?php echo htmlspecialchars($endpoint_verify); ?></code></pre>
        <h4>Request Body (JSON)</h4>
        <pre><code>{
    "unique_payment_id": "a1b2c3...", // The original trx_id from the payment_url
    "order_id": "YOUR_UNIQUE_ORDER_ID_123" // Your internal order ID
}</code></pre>
        <h4>Success Response (JSON)</h4>
        <pre><code>{
    "success": true,
    "message": "Payment has been successfully verified.",
    "transaction_id": "CJH1D2Y4W1"
}</code></pre>
        <p>You can then compare the `transaction_id` from this response with the one you received in the user's browser URL to be 100% certain of the payment's authenticity.</p>
    </div>

</div>

</div> <!-- .main-content -->
<script src="js/main.js?v=<?php echo filemtime('js/main.js'); ?>"></script>
</body>
</html>