<?php
/**
 * API ENDPOINT for Creating a Gateway
 * Version 3.0 - ADVANCED DIAGNOSTIC & DEBUGGING MODE
 * This version will log every detail of the incoming request to a file named 'api_debug.log'.
 */

// ★★★ ডিবাগিং এর জন্য লগ ফাইল তৈরি ★★★
$log_file = __DIR__ . '/api_debug.log';

// ফাংশন: লগ ফাইলে তথ্য লেখার জন্য
function write_to_log($message) {
    global $log_file;
    // লগ মেসেজের সাথে তারিখ ও সময় যুক্ত করা হচ্ছে
    $timestamp = date('Y-m-d H:i:s');
    file_put_contents($log_file, "[{$timestamp}] " . $message . "\n\n", FILE_APPEND);
}

// --- ডায়াগনস্টিকস শুরু ---
$headers = getallheaders();
$raw_post_data = file_get_contents('php://input');

$log_message = "--- NEW REQUEST RECEIVED ---\n";
$log_message .= "[METHOD] " . $_SERVER['REQUEST_METHOD'] . "\n";
$log_message .= "[HEADERS] " . print_r($headers, true) . "\n";
$log_message .= "[POST DATA] " . print_r($_POST, true) . "\n";
$log_message .= "[RAW BODY] " . $raw_post_data . "\n";

write_to_log($log_message);
// --- ডায়াগনস্টিকস শেষ ---


header('Content-Type: application/json');
require '../db_connect.php';

try {
    // ধাপ ১: হেডার থেকে API Key গ্রহণ করা এবং লগ করা
    $api_key = $headers['X-API-KEY'] ?? $headers['x-api-key'] ?? null; // Case-insensitive check

    if (empty($api_key)) {
        $error_msg = 'Authentication failed: X-API-KEY header is completely missing from the request.';
        write_to_log("[API KEY CHECK] FAILED: " . $error_msg);
        http_response_code(401);
        echo json_encode(['success' => false, 'message' => $error_msg]);
        exit();
    }
    
    write_to_log("[API KEY CHECK] SUCCESS: API Key found in header. Value: " . $api_key);

    // ধাপ ২: API Key ব্যবহার করে ডাটাবেস থেকে user_id খুঁজে বের করা এবং লগ করা
    $stmt_user = $pdo->prepare("SELECT user_id FROM settings WHERE api_key = ?");
    $stmt_user->execute([$api_key]);
    $setting = $stmt_user->fetch(PDO::FETCH_ASSOC);

    if (!$setting) {
        $error_msg = 'Authentication failed: The provided API Key is invalid and was not found in the database.';
        write_to_log("[DATABASE LOOKUP] FAILED: " . $error_msg);
        http_response_code(403);
        echo json_encode(['success' => false, 'message' => $error_msg]);
        exit();
    }
    
    $user_id = $setting['user_id'];
    write_to_log("[DATABASE LOOKUP] SUCCESS: User ID found for the API Key. User ID: " . $user_id);

    // ধাপ ৩: অ্যাপ থেকে পাঠানো ডেটা গ্রহণ করা
    $gateway_key = $_POST['gateway_key'] ?? '';

    if (empty($gateway_key)) {
        $error_msg = 'Required field missing: gateway_key.';
        write_to_log("[INPUT VALIDATION] FAILED: " . $error_msg);
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => $error_msg]);
        exit();
    }

    // গেটওয়ে কী-এর উপর ভিত্তি করে একটি ডিফল্ট নাম সেট করা
    $display_name = '';
    switch ($gateway_key) {
        case 'bkash_personal': $display_name = 'Bkash Personal'; break;
        case 'nagad_personal': $display_name = 'Nagad Personal'; break;
        case 'rocket_personal': $display_name = 'Rocket Personal'; break;
        case 'upay_personal': $display_name = 'Upay Personal'; break;
        case 'bkash_merchant_tokenized': $display_name = 'bKash Marchant (Tokenized)'; break;
        case 'nagad_merchant': $display_name = 'Nagad Marchant'; break;
        case 'bkash_merchant_checkout': $display_name = 'bKash Marchant (Checkout)'; break;
        default:
            $error_msg = 'Invalid gateway_key provided.';
            write_to_log("[INPUT VALIDATION] FAILED: " . $error_msg);
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => $error_msg]);
            exit();
    }

    // ধাপ ৪: ডাটাবেসে নতুন গেটওয়ে ইনসার্ট করা
    write_to_log("[DATABASE INSERT] Attempting to insert gateway with User ID: " . $user_id);
    
    $sql = "INSERT INTO gateways (user_id, display_name, gateway_key) VALUES (:user_id, :display_name, :gateway_key)";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([':user_id' => $user_id, ':display_name' => $display_name, ':gateway_key' => $gateway_key]);
        
    if ($stmt->rowCount() > 0) {
        write_to_log("[DATABASE INSERT] SUCCESS: Gateway created successfully.");
        echo json_encode(['success' => true, 'message' => 'Gateway created successfully.']);
    } else {
        $error_msg = 'Failed to create the gateway in the database, although the request was valid.';
        write_to_log("[DATABASE INSERT] FAILED: " . $error_msg);
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => $error_msg]);
    }

} catch (PDOException $e) {
    $error_msg = 'Database Error: ' . $e->getMessage();
    write_to_log("[CRITICAL ERROR] " . $error_msg);
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => $error_msg]);
}
?>