<?php
header('Content-Type: application/json');
require '../db_connect.php';

/**
 * Fee Deduction Logic Function
 * This function handles the logic of deducting credits from a user's account.
 */
function deduct_fee_and_log($pdo, $user_id, $transaction_db_id_from_product) {
    // Step 1: Calculate total available credits for the user
    $free_credits = 50; // All new users get 50 free credits
    $stmt_purchased = $pdo->prepare("SELECT COALESCE(SUM(amount_bdt), 0) as total_purchased FROM user_funds WHERE user_id = ? AND status = 'completed' AND transaction_unique_id LIKE 'manual_purchase_%'");
    $stmt_purchased->execute([$user_id]);
    $purchased_credits = $stmt_purchased->fetchColumn();
    $total_available_credits = $free_credits + $purchased_credits;

    // Step 2: Calculate used credits so far, locking the row to prevent race conditions
    $stmt_used = $pdo->prepare("SELECT COUNT(id) FROM credit_transactions WHERE user_id = ? FOR UPDATE");
    $stmt_used->execute([$user_id]);
    $used_credits = $stmt_used->fetchColumn();

    // Step 3: Check if the user has enough credits
    if ($used_credits >= $total_available_credits) {
        // Log the error but don't throw an exception that would fail the API call,
        // as the main transaction in product_project was already successful.
        error_log("Insufficient credits for user_id: {$user_id}. Fee deduction skipped for product transaction ID: {$transaction_db_id_from_product}");
        return false; // Indicate failure
    }

    // Step 4: Define fee and log the credit usage in the credit_transactions table
    $fee_bdt = '1.00';
    $fee_usd = bcdiv($fee_bdt, '120', 4); // Use bcdiv for precision
    
    // The payment_transaction_id can be used to reference the transaction from the product_project for auditing purposes.
    $stmt_log = $pdo->prepare(
        "INSERT INTO credit_transactions (user_id, payment_transaction_id, cost_bdt, cost_usd, credits_deducted, description)
         VALUES (?, ?, ?, ?, 1, 'API Payment Verification Fee')"
    );
    // Storing the product project's transaction ID for reference
    $stmt_log->execute([$user_id, $transaction_db_id_from_product, $fee_bdt, $fee_usd]);
    
    return true; // Indicate success
}

// --- Main API Logic ---
try {
    // Authenticate the request using the API Key from the header
    $api_key = $_SERVER['HTTP_X_API_KEY'] ?? '';
    if (empty($api_key)) {
        throw new Exception("API Key is missing.", 401);
    }

    $stmt_user = $pdo->prepare("SELECT user_id FROM settings WHERE api_key = ?");
    $stmt_user->execute([$api_key]);
    $setting = $stmt_user->fetch(PDO::FETCH_ASSOC);
    if (!$setting) {
        throw new Exception("Invalid API Key.", 403);
    }
    $user_id = $setting['user_id'];

    // Get the local transaction ID sent from the product_project
    $transaction_db_id = $_POST['transaction_db_id'] ?? null;
    if (!$transaction_db_id) {
        throw new Exception("Transaction DB ID from product project is required.", 400);
    }
    
    // Execute the fee deduction within a database transaction for data integrity
    $pdo->beginTransaction();
    deduct_fee_and_log($pdo, $user_id, $transaction_db_id);
    $pdo->commit();
    
    // Send a success response
    echo json_encode(['success' => true, 'message' => 'Fee logged successfully.']);

} catch (Exception $e) {
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    $code = is_int($e->getCode()) && $e->getCode() !== 0 ? $e->getCode() : 500;
    http_response_code($code);
    echo json_encode(['success' => false, 'message' => 'Central Server Error: ' . $e->getMessage()]);
}
?>