<?php
/**
 * API Endpoint for Purchasing Credits from Account Balance
 *
 * This script is designed to be robust and handle requests from two different sources:
 * 1. The native Android application, which authenticates using an API Key in the HTTP headers.
 * 2. The web-based dashboard, which authenticates using a standard PHP session.
 *
 * It processes the credit purchase, deducts the cost from the user's balance,
 * logs the transaction, and returns a context-aware success or error message in JSON format.
 *
 * @version 3.0 (Production Ready - Complete & Unabridged)
 */

// Step 1: Set the content type to JSON for all responses to ensure consistency.
header('Content-Type: application/json');

// Step 2: Include the database connection file. This is essential for all database operations.
require '../db_connect.php';

// Step 3: Start a PHP session if one is not already active.
// This is crucial for authenticating requests coming from the web dashboard.
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Initialize variables that will be determined by the authentication process.
$user_id = null;
$request_source = 'unknown'; // This will help in sending the correct success message.

// --- Step 4: Dual Authentication Logic ---
// This is the core logic to identify the user from either the app or the web.

// First, check for an API key in the request headers. This is the method used by the Android app.
if (isset($_SERVER['HTTP_X_API_KEY']) && !empty($_SERVER['HTTP_X_API_KEY'])) {
    
    $api_key = $_SERVER['HTTP_X_API_KEY'];
    
    // Find the user associated with the provided API key from the 'settings' table.
    $stmt_user_lookup = $pdo->prepare("SELECT user_id FROM settings WHERE api_key = ?");
    $stmt_user_lookup->execute([$api_key]);
    $setting = $stmt_user_lookup->fetch(PDO::FETCH_ASSOC);
    
    if ($setting) {
        $user_id = $setting['user_id'];
        $request_source = 'app'; // Mark the request source as 'app'.
    } else {
        // If the API key is provided but is not valid, deny access.
        http_response_code(403); // 403 Forbidden
        die(json_encode(['success' => false, 'message' => 'Invalid API Key provided. Authentication failed.']));
    }
} 
// If no API key is found, check if a user is logged into the web dashboard via a PHP session.
else if (isset($_SESSION['user_id'])) {
    
    $user_id = $_SESSION['user_id'];
    $request_source = 'web'; // Mark the request source as 'web'.
}

// If after both checks, no user could be identified, deny access.
if ($user_id === null) {
    http_response_code(401); // 401 Unauthorized
    die(json_encode(['success' => false, 'message' => 'Authentication failed. An API Key or an active user session is required.']));
}

// --- Step 5: Data Input Handling ---
// This section retrieves the number of credits to purchase from either a web form or a JSON request.

// Define the conversion rate.
define('USD_TO_BDT_RATE', 120);

$credits_to_purchase = 0;

// Check for data from a standard web form (x-www-form-urlencoded).
if (!empty($_POST['credits'])) {
    $credits_to_purchase = (int)$_POST['credits'];
} 
// If not found, check for data from a JSON request body (raw).
else {
    $json_data = file_get_contents('php://input');
    $data = json_decode($json_data, true);
    if (isset($data['credits'])) {
        $credits_to_purchase = (int)$data['credits'];
    }
}

// Validate that the number of credits is a positive number.
if ($credits_to_purchase <= 0) {
    http_response_code(400); // 400 Bad Request
    echo json_encode(['success' => false, 'message' => 'Please provide a valid, positive number of credits to purchase.']);
    exit();
}

// Calculate the cost in both BDT and USD.
$cost_bdt = $credits_to_purchase; // Business logic: 1 Credit = 1 BDT
$cost_usd = $cost_bdt / USD_TO_BDT_RATE;

// --- Step 6: Database Transaction Logic ---
// This is a critical section. Using a transaction ensures data integrity.

try {
    // Start a database transaction. All subsequent queries must succeed, or all will be rolled back.
    $pdo->beginTransaction();

    // Get the user's current balance. 'FOR UPDATE' locks the row to prevent other simultaneous transactions from reading stale data.
    $stmt_user = $pdo->prepare("SELECT balance FROM users WHERE id = ? FOR UPDATE");
    $stmt_user->execute([$user_id]);
    $user = $stmt_user->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
        throw new Exception("User account not found in the database.");
    }
    if ($user['balance'] < $cost_usd) {
        throw new Exception("Insufficient balance to complete the purchase. Please add funds to your account first.");
    }

    // Deduct the cost from the user's main balance.
    $stmt_deduct = $pdo->prepare("UPDATE users SET balance = balance - ? WHERE id = ?");
    $stmt_deduct->execute([$cost_usd, $user_id]);

    // Create a unique transaction ID for logging this specific purchase.
    $transaction_unique_id = 'manual_purchase_' . time() . '_' . bin2hex(random_bytes(4));
    
    // Log the transaction in the `user_funds` table. This is how the system knows how many credits were purchased.
    $stmt_add_fund = $pdo->prepare(
        "INSERT INTO user_funds (user_id, transaction_unique_id, amount_usd, amount_bdt, status) 
         VALUES (?, ?, ?, ?, 'completed')"
    );
    $stmt_add_fund->execute([$user_id, $transaction_unique_id, $cost_usd, $cost_bdt]);
    
    // If all database operations were successful, commit the changes permanently.
    $pdo->commit();

    // --- Step 7: Send Context-Aware Success Response ---
    
    $success_message = '';
    if ($request_source === 'app') {
        // A simple, clear message for the Android app.
        $success_message = 'Credits purchased successfully!';
    } else {
        // A message indicating a page reload for the web dashboard.
        $success_message = 'Credits purchased successfully. The page will now reload.';
    }

    echo json_encode(['success' => true, 'message' => $success_message]);

} catch (Exception $e) {
    // If any error occurred during the 'try' block, roll back all database changes.
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    // Send a user-friendly error message in a JSON response.
    http_response_code(400); // 400 Bad Request is appropriate for business logic failures (e.g., insufficient funds).
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
?>