<?php
header('Content-Type: application/json');
require 'auth_check.php';
require 'db_connect.php';

// --- Configuration ---
define('USD_TO_BDT_RATE', 120);

// --- Input Processing ---
$json_data = file_get_contents('php://input');
$data = json_decode($json_data, true);
$amount_usd = (float)($data['amount'] ?? 0);

if ($amount_usd <= 0) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Invalid amount provided.']);
    exit();
}

// --- Data Preparation ---
$user_id = $_SESSION['user_id'];
$amount_bdt = $amount_usd * USD_TO_BDT_RATE;
$transaction_unique_id = sha1(uniqid(rand(), true) . microtime());

// --- Base URL for Callbacks ---
$protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? "https" : "http";
$host = $_SERVER['HTTP_HOST'];
$directory = rtrim(dirname($_SERVER['PHP_SELF']), '/\\');
$base_project_url = "{$protocol}://{$host}{$directory}";

// Success URL points back to billing page with success flags
$success_url = "{$base_project_url}/billing.php?funding_status=success&trx_id=" . $transaction_unique_id;
$cancel_url = "{$base_project_url}/billing.php?funding_status=cancelled";

try {
    $pdo->beginTransaction();

    // Step 1: Log the funding attempt in the new `user_funds` table
    $stmt_funds = $pdo->prepare(
        "INSERT INTO user_funds (user_id, transaction_unique_id, amount_usd, amount_bdt, status) 
         VALUES (?, ?, ?, ?, 'pending')"
    );
    $stmt_funds->execute([$user_id, $transaction_unique_id, $amount_usd, $amount_bdt]);

    // Step 2: Create a payment session in the main `transactions` table
    $stmt_trans = $pdo->prepare(
        "INSERT INTO transactions (user_id, unique_id, amount, customer_name, customer_email, success_url, cancel_url, meta_data) 
         VALUES (?, ?, ?, ?, ?, ?, ?, ?)"
    );
    $stmt_trans->execute([
        $user_id,
        $transaction_unique_id,
        $amount_bdt, // Use BDT amount for the payment gateway
        $_SESSION['username'] ?? 'User',
        $_SESSION['user_email'] ?? 'email@example.com', // Assuming you store email in session
        $success_url,
        $cancel_url,
        json_encode(['source' => 'user_funding', 'amount_usd' => $amount_usd])
    ]);

    $pdo->commit();

    // Step 3: Respond with the checkout URL
    $payment_url = "{$base_project_url}/checkout.php?trx_id={$transaction_unique_id}";
    echo json_encode(['success' => true, 'payment_url' => $payment_url]);

} catch (PDOException $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Failed to initiate funding session.', 'error' => $e->getMessage()]);
}
