<?php
require 'auth_check.php';
require 'db_connect.php';

// --- AJAX অনুরোধ হ্যান্ডেল করার জন্য PHP কোড ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // JSON রেসপন্স পাঠানোর জন্য হেডার সেট করা
    header('Content-Type: application/json');

    $response = ['success' => false, 'message' => 'An unknown error occurred.'];
    
    // ★★★ নতুন পরিবর্তন: আপডেট টাইপ (automatic/manual) গ্রহণ করা হচ্ছে ★★★
    $version = trim($_POST['version']);
    $update_type = $_POST['update_type'] ?? 'automatic'; // ডিফল্ট 'automatic'
    $sql_migration = trim($_POST['sql_migration']);
    $zip_file = $_FILES['zip_file'];

    if (!empty($version) && isset($zip_file) && $zip_file['error'] === UPLOAD_ERR_OK) {
        $upload_dir = 'updates/';
        if (!is_dir($upload_dir)) {
            if (!mkdir($upload_dir, 0777, true)) {
                $response['message'] = 'Failed to create updates directory. Please check permissions.';
                echo json_encode($response);
                exit();
            }
        }

        $file_name = "update-v{$version}.zip";
        $destination = $upload_dir . $file_name;

        if (move_uploaded_file($zip_file['tmp_name'], $destination)) {
            try {
                $files_manifest = [];
                $dirs_manifest = [];
                
                $zip = new ZipArchive;
                if ($zip->open($destination) === TRUE) {
                    for ($i = 0; $i < $zip->numFiles; $i++) {
                        $path = $zip->getNameIndex($i);
                        if (substr($path, -1) === '/') {
                            $dirs_manifest[] = $path;
                        } else {
                            $files_manifest[$path] = md5($zip->getFromIndex($i));
                        }
                    }
                    $zip->close();
                    
                    $manifest_content = ['files' => $files_manifest, 'dirs' => $dirs_manifest];
                    $manifest_path = $upload_dir . "update-v{$version}_manifest.json";
                    file_put_contents($manifest_path, json_encode($manifest_content, JSON_PRETTY_PRINT));
                } else {
                    throw new Exception("Could not open ZIP file to generate manifest.");
                }

                // ★★★ নতুন পরিবর্তন: ডাটাবেসে update_type সহ তথ্য সংরক্ষণ করা হচ্ছে ★★★
                $stmt = $pdo->prepare(
                    "INSERT INTO updates (version, zip_path, sql_migration, update_type) VALUES (?, ?, ?, ?)"
                );
                $stmt->execute([$version, $destination, $sql_migration ?: null, $update_type]);
                
                $response['success'] = true;
                $response['message'] = "Version {$version} uploaded successfully! Update type: {$update_type}.";

            } catch (Exception $e) {
                $response['message'] = "Error: " . $e->getMessage();
                if (file_exists($destination)) unlink($destination);
                if (isset($manifest_path) && file_exists($manifest_path)) unlink($manifest_path);
            }
        } else {
            $response['message'] = 'Failed to move uploaded file.';
        }
    } else {
        $response['message'] = 'Version number and a valid ZIP file are required.';
    }

    echo json_encode($response);
    exit();
}

$updates = $pdo->query("SELECT * FROM updates ORDER BY id DESC")->fetchAll(PDO::FETCH_ASSOC);

include 'templates/header.php';
include 'templates/sidebar.php';
?>

<style>
    .progress-bar-container { display: none; width: 100%; background-color: #f3f3f3; border: 1px solid #ccc; border-radius: 8px; margin-top: 15px; overflow: hidden; }
    .progress-bar { width: 0%; height: 25px; background-color: #0d6efd; border-radius: 8px; text-align: center; line-height: 25px; color: white; font-weight: bold; transition: width 0.3s ease-in-out; }
    .message-area { margin-top: 20px; padding: 15px; border-radius: 8px; display: none; }
    .message-area.success { background-color: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
    .message-area.error { background-color: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
    .update-type-options { display: flex; gap: 20px; margin-bottom: 15px; }
</style>

<div class="page-container">
    <div class="page-header"><h1 class="page-title">Application Updates</h1></div>
    <div id="message-area" class="message-area"></div>

    <div class="content-card">
        <h3>Upload New Version</h3>
        <form id="uploadForm" action="updates.php" method="POST" enctype="multipart/form-data">
            <div class="form-group">
                <label for="version">Version Number (e.g., 1.0.1)</label>
                <input type="text" id="version" name="version" class="form-control" required>
            </div>
            
            <!-- ★★★ নতুন পরিবর্তন: আপডেট টাইপ সিলেক্ট করার অপশন ★★★ -->
            <div class="form-group">
                <label>Update Type</label>
                <div class="update-type-options">
                    <label><input type="radio" name="update_type" value="automatic" checked> Automatic</label>
                    <label><input type="radio" name="update_type" value="manual"> Manual</label>
                </div>
                <small>Automatic updates will be installed by the client's cron job. Manual updates will require the client to click 'Install'.</small>
            </div>

            <div class="form-group">
                <label for="zip_file">Update ZIP File</label>
                <input type="file" id="zip_file" name="zip_file" class="form-control" accept=".zip" required>
            </div>
            <div class="form-group">
                <label for="sql_migration">Database Migration SQL (Optional)</label>
                <textarea id="sql_migration" name="sql_migration" rows="8" class="form-control" style="width: 100%; font-family: monospace;"></textarea>
                <small>Enter any SQL commands needed for this update. E.g., ALTER TABLE, CREATE TABLE, etc.</small>
            </div>
            <div class="progress-bar-container" id="progress-container"><div class="progress-bar" id="progress-bar">0%</div></div>
            <div class="form-actions"><button type="submit" class="btn btn-primary" id="uploadBtn">Upload Update</button></div>
        </form>
    </div>

    <div class="content-card" style="margin-top: 20px;">
        <h3>Uploaded Versions</h3>
        <table class="data-table">
            <thead><tr><th>Version</th><th>Update Type</th><th>File Path</th><th>Has DB Migration?</th><th>Upload Date</th></tr></thead>
            <tbody id="updates-table-body">
                <?php foreach($updates as $update): ?>
                <tr>
                    <td><?php echo htmlspecialchars($update['version']); ?></td>
                    <td><?php echo htmlspecialchars(ucfirst($update['update_type'])); ?></td>
                    <td><?php echo htmlspecialchars($update['zip_path']); ?></td>
                    <td><?php echo !empty($update['sql_migration']) ? 'Yes' : 'No'; ?></td>
                    <td><?php echo date("d M, Y H:i", strtotime($update['created_at'])); ?></td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const uploadForm = document.getElementById('uploadForm');
    uploadForm.addEventListener('submit', function(event) {
        event.preventDefault();
        const formData = new FormData(uploadForm);
        const progressBar = document.getElementById('progress-bar');
        const progressContainer = document.getElementById('progress-container');
        const messageArea = document.getElementById('message-area');
        const uploadBtn = document.getElementById('uploadBtn');

        progressContainer.style.display = 'block';
        uploadBtn.disabled = true;
        uploadBtn.textContent = 'Uploading...';

        const xhr = new XMLHttpRequest();
        xhr.upload.addEventListener('progress', function(e) {
            if (e.lengthComputable) {
                const percentComplete = Math.round((e.loaded / e.total) * 100);
                progressBar.style.width = percentComplete + '%';
                progressBar.textContent = percentComplete + '%';
            }
        });
        xhr.onload = function() {
            uploadBtn.disabled = false;
            uploadBtn.textContent = 'Upload Update';
            if (xhr.status === 200) {
                const response = JSON.parse(xhr.responseText);
                showMessage(response.message, response.success);
                if (response.success) {
                    setTimeout(() => { window.location.reload(); }, 2000);
                }
            } else {
                showMessage('An error occurred during upload. Status: ' + xhr.status, false);
            }
        };
        xhr.onerror = function() {
            showMessage('A network error occurred.', false);
            uploadBtn.disabled = false;
            uploadBtn.textContent = 'Upload Update';
        };
        xhr.open('POST', 'updates.php', true);
        xhr.send(formData);
    });
    function showMessage(message, isSuccess) {
        const messageArea = document.getElementById('message-area');
        messageArea.textContent = message;
        messageArea.className = 'message-area ' + (isSuccess ? 'success' : 'error');
        messageArea.style.display = 'block';
    }
});
</script>