<?php
require 'auth_check.php';
require 'db_connect.php';

include 'templates/header.php';
include 'templates/sidebar.php';

$user_id = $_SESSION['user_id'];
$customers = [];

try {
    $sql = "
        SELECT 
            t1.customer_name, 
            t1.customer_email,
            t1.phone,
            t1.address,
            t1.city,
            t1.state,
            t1.postcode,
            t1.country
        FROM transactions t1
        INNER JOIN (
            SELECT customer_email, MAX(id) as max_id
            FROM transactions
            WHERE user_id = :user_id AND customer_email IS NOT NULL AND customer_email != ''
            GROUP BY customer_email
        ) t2 ON t1.id = t2.max_id AND t1.customer_email = t2.customer_email
        ORDER BY t1.id DESC
    ";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([':user_id' => $user_id]);
    $customers = $stmt->fetchAll(PDO::FETCH_ASSOC);

} catch (PDOException $e) {
    error_log("Customer data fetch error: " . $e->getMessage());
}

?>

<div class="page-container">
    <div class="page-header">
        <div>
            <p class="breadcrumbs">Customers &gt; List</p>
            <h1 class="page-title">Customers</h1>
        </div>
        <div class="page-header-actions">
            <button class="btn btn-secondary">
                Actions
            </button>
            <a href="#" id="newCustomerBtn" class="btn btn-primary">
                <i class="fas fa-plus"></i> New customer
            </a>
        </div>
    </div>

    <div class="content-card">
        <div class="card-toolbar">
            <div class="toolbar-left">
                <i class="fas fa-filter icon-btn"></i> 0
            </div>
            <div class="toolbar-right" style="display: flex; align-items: center; gap: 15px;">
                <div class="search-box">
                    <i class="fas fa-search"></i>
                    <input type="text" placeholder="Search">
                </div>
                <i class="fas fa-bars icon-btn"></i>
            </div>
        </div>

        <table class="data-table">
            <thead>
                <tr>
                    <th><input type="checkbox"></th>
                    <th>Name <i class="fas fa-chevron-down"></i></th>
                    <th>Email <i class="fas fa-chevron-down"></i></th>
                    <th>Phone <i class="fas fa-chevron-down"></i></th>
                    <th></th>
                </tr>
            </thead>
            <tbody>
                <?php if (count($customers) > 0): ?>
                    <?php foreach ($customers as $customer): ?>
                        <tr>
                            <td><input type="checkbox"></td>
                            <td><?php echo htmlspecialchars($customer['customer_name']); ?></td>
                            <td><?php echo htmlspecialchars($customer['customer_email']); ?></td>
                            <td><?php echo htmlspecialchars($customer['phone'] ?: 'N/A'); ?></td>
                            <td class="actions" style="text-align: right;">
                                <a href="#" class="action-edit edit-btn" 
                                   data-name="<?php echo htmlspecialchars($customer['customer_name']); ?>" 
                                   data-email="<?php echo htmlspecialchars($customer['customer_email']); ?>" 
                                   data-phone="<?php echo htmlspecialchars($customer['phone'] ?: ''); ?>"
                                   data-address="<?php echo htmlspecialchars($customer['address'] ?: ''); ?>"
                                   data-city="<?php echo htmlspecialchars($customer['city'] ?: ''); ?>"
                                   data-state="<?php echo htmlspecialchars($customer['state'] ?: ''); ?>"
                                   data-postcode="<?php echo htmlspecialchars($customer['postcode'] ?: ''); ?>"
                                   data-country="<?php echo htmlspecialchars($customer['country'] ?: ''); ?>">
                                   <i class="fas fa-pencil-alt"></i> Edit
                                </a>
                                <a href="delete_customer.php?email=<?php echo urlencode($customer['customer_email']); ?>" class="action-delete delete-btn">
                                   <i class="fas fa-trash-alt"></i> Delete
                                </a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php else: ?>
                    <tr>
                        <td colspan="5" style="text-align:center; padding: 20px;">No customers found.</td>
                    </tr>
                <?php endif; ?>
            </tbody>
        </table>
        
        <div class="card-footer">
            <div class="pagination-controls">
                <span>Per page</span>
                <select><option selected>10</option><option>20</option><option>50</option></select>
            </div>
        </div>
    </div>
</div>

<!-- Add/Edit Customer Modal -->
<div id="customerModal" class="modal">
    <div class="modal-content" style="max-width: 800px;">
        <form id="customerForm" action="add_customer.php" method="POST">
            <input type="hidden" name="original_email" id="original_email">
            <div class="modal-header">
                <h2 id="modalTitle">Create customer</h2>
                <span class="close-button">&times;</span>
            </div>
            <div class="modal-body" style="padding: 20px 0;">
                <div class="form-grid" style="grid-template-columns: repeat(3, 1fr); gap: 20px;">
                    <div class="form-group"><label for="customer_name">Name<span class="required-star">*</span></label><input type="text" id="customer_name" name="name" required></div>
                    <div class="form-group"><label for="customer_email">Email<span class="required-star">*</span></label><input type="email" id="customer_email" name="email" required></div>
                    <div class="form-group"><label for="customer_phone">Phone<span class="required-star">*</span></label><input type="text" id="customer_phone" name="phone" required></div>
                    <div class="form-group"><label for="address">Address</label><input type="text" id="address" name="address"></div>
                    <div class="form-group"><label for="city">City</label><input type="text" id="city" name="city"></div>
                    <div class="form-group"><label for="state">State</label><input type="text" id="state" name="state"></div>
                    <div class="form-group"><label for="postcode">Postcode</label><input type="text" id="postcode" name="postcode"></div>
                    <div class="form-group"><label for="country">Country</label>
                        <div class="select-wrapper"><select id="country" name="country"><option value="">Select an option</option><option value="BD">Bangladesh</option><option value="US">United States</option></select></div>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary cancel-btn">Cancel</button>
                <button type="submit" class="btn btn-primary">Create</button>
            </div>
        </form>
    </div>
</div>


</div> <!-- .main-content 닫는 태그 -->

<script src="js/main.js?v=<?php echo filemtime('js/main.js'); ?>"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    const modal = document.getElementById('customerModal');
    const newCustomerBtn = document.getElementById('newCustomerBtn');
    const closeBtn = modal.querySelector('.close-button');
    const cancelBtn = modal.querySelector('.cancel-btn');
    const form = document.getElementById('customerForm');
    const modalTitle = document.getElementById('modalTitle');

    const openModal = () => modal.style.display = 'block';
    const closeModal = () => modal.style.display = 'none';

    newCustomerBtn.addEventListener('click', function(e) {
        e.preventDefault();
        form.reset();
        form.action = 'add_customer.php';
        modalTitle.innerText = 'Create customer';
        document.getElementById('original_email').value = '';
        openModal();
    });
    
    document.querySelectorAll('.edit-btn').forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            form.reset();
            modalTitle.innerText = 'Edit customer';
            form.action = 'edit_customer.php';
            
            document.getElementById('customer_name').value = this.dataset.name;
            document.getElementById('customer_email').value = this.dataset.email;
            document.getElementById('customer_phone').value = this.dataset.phone;
            document.getElementById('original_email').value = this.dataset.email;
            document.getElementById('address').value = this.dataset.address;
            document.getElementById('city').value = this.dataset.city;
            document.getElementById('state').value = this.dataset.state;
            document.getElementById('postcode').value = this.dataset.postcode;
            document.getElementById('country').value = this.dataset.country;
            
            openModal();
        });
    });

    document.querySelectorAll('.delete-btn').forEach(button => {
        button.addEventListener('click', function(e) {
            if (!confirm('Are you sure you want to delete this customer and all their records?')) {
                e.preventDefault();
            }
        });
    });

    closeBtn.addEventListener('click', closeModal);
    cancelBtn.addEventListener('click', closeModal);
    window.addEventListener('click', function(event) {
        if (event.target == modal) {
            closeModal();
        }
    });
});
</script>
</body>
</html>
