<?php
/**
 * API Endpoint for Purchasing Credits from Account Balance
 * Version 3.1 - Now uses BCMath for arbitrary-precision mathematics.
 */

header('Content-Type: application/json');
require '../db_connect.php';

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

$user_id = null;
$request_source = 'unknown';

// Dual Authentication Logic (API Key or Session)
if (isset($_SERVER['HTTP_X_API_KEY']) && !empty($_SERVER['HTTP_X_API_KEY'])) {
    $api_key = $_SERVER['HTTP_X_API_KEY'];
    $stmt_user_lookup = $pdo->prepare("SELECT user_id FROM settings WHERE api_key = ?");
    $stmt_user_lookup->execute([$api_key]);
    $setting = $stmt_user_lookup->fetch(PDO::FETCH_ASSOC);
    if ($setting) {
        $user_id = $setting['user_id'];
        $request_source = 'app';
    } else {
        http_response_code(403);
        die(json_encode(['success' => false, 'message' => 'Invalid API Key provided. Authentication failed.']));
    }
} else if (isset($_SESSION['user_id'])) {
    $user_id = $_SESSION['user_id'];
    $request_source = 'web';
}

if ($user_id === null) {
    http_response_code(401);
    die(json_encode(['success' => false, 'message' => 'Authentication failed. An API Key or an active user session is required.']));
}

// ★★★ পরিবর্তন: রেট এবং ইনপুট স্ট্রিং হিসেবে গ্রহণ করা হচ্ছে ★★★
define('USD_TO_BDT_RATE', '120');

$credits_to_purchase = '0';
if (!empty($_POST['credits'])) {
    $credits_to_purchase = (string)(int)$_POST['credits'];
} else {
    $json_data = file_get_contents('php://input');
    $data = json_decode($json_data, true);
    if (isset($data['credits'])) {
        $credits_to_purchase = (string)(int)$data['credits'];
    }
}

// BCMath দিয়ে ইনপুট ভ্যালিডেশন
if (bccomp($credits_to_purchase, '0', 0) <= 0) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Please provide a valid, positive number of credits to purchase.']);
    exit();
}

$cost_bdt = $credits_to_purchase; // Business logic: 1 Credit = 1 BDT

// ★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★
// ★★★      BCMath দিয়ে খরচ (Cost) গণনা       ★★★
// ★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★
// সাধারণ ভাগ (/) এর পরিবর্তে bcdiv() ব্যবহার করে নির্ভুল হিসাব করা হয়েছে।
// শেষের '4' মানে হলো দশমিকের পর চার ঘর পর্যন্ত নির্ভুলতা রাখা হবে।
$cost_usd = bcdiv($cost_bdt, USD_TO_BDT_RATE, 4);

try {
    $pdo->beginTransaction();

    $stmt_user = $pdo->prepare("SELECT balance FROM users WHERE id = ? FOR UPDATE");
    $stmt_user->execute([$user_id]);
    $user = $stmt_user->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
        throw new Exception("User account not found in the database.");
    }
    
    // ★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★
    // ★★★   BCMath দিয়ে ব্যালেন্স যাচাই ও আপডেট   ★★★
    // ★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★
    
    // bccomp() দিয়ে নির্ভুলভাবে ব্যালেন্স তুলনা করা হচ্ছে।
    // যদি user balance, cost_usd এর চেয়ে কম হয়, তাহলে bccomp -1 রিটার্ন করবে।
    if (bccomp((string)$user['balance'], $cost_usd, 4) < 0) {
        throw new Exception("Insufficient balance to complete the purchase. Please add funds to your account first.");
    }

    // bcsub() দিয়ে নির্ভুলভাবে নতুন ব্যালেন্স গণনা করা হচ্ছে।
    $new_balance = bcsub((string)$user['balance'], $cost_usd, 4);

    // সরাসরি SQL-এ বিয়োগ না করে, PHP-তে গণনা করা নতুন ব্যালেন্সটি আপডেট করা হচ্ছে।
    // এটি Race Condition প্রতিরোধে সাহায্য করে এবং হিসাবকে নির্ভুল রাখে।
    $stmt_deduct = $pdo->prepare("UPDATE users SET balance = ? WHERE id = ?");
    $stmt_deduct->execute([$new_balance, $user_id]);
    
    // ★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★

    $transaction_unique_id = 'manual_purchase_' . time() . '_' . bin2hex(random_bytes(4));
    
    $stmt_add_fund = $pdo->prepare(
        "INSERT INTO user_funds (user_id, transaction_unique_id, amount_usd, amount_bdt, status) 
         VALUES (?, ?, ?, ?, 'completed')"
    );
    $stmt_add_fund->execute([$user_id, $transaction_unique_id, $cost_usd, $cost_bdt]);
    
    $pdo->commit();

    $success_message = ($request_source === 'app') 
        ? 'Credits purchased successfully!' 
        : 'Credits purchased successfully. The page will now reload.';

    echo json_encode(['success' => true, 'message' => $success_message]);

} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
?>