<?php
// updater_logic.php (v2.2 - Final, Unabridged Version)

/**
 * cURL ব্যবহার করে একটি ফাইল ডাউনলোড করার জন্য Helper Function.
 * এটি file_get_contents-এর চেয়ে অনেক বেশি নির্ভরযোগ্য।
 * @param string $url যে URL থেকে ফাইল ডাউনলোড করা হবে।
 * @param string $save_path সার্ভারে ফাইলটি কোথায় সেভ করা হবে।
 * @return bool সফল হলে true, ব্যর্থ হলে Exception throw করবে।
 * @throws Exception যদি ফাইল লেখা বা ডাউনলোড করা সম্ভব না হয়।
 */
function download_file_with_curl($url, $save_path) {
    // ফাইল লেখার জন্য খোলা হচ্ছে
    $file_pointer = fopen($save_path, 'w');
    if (!$file_pointer) {
        throw new Exception("Could not open file for writing: " . $save_path);
    }

    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_FILE, $file_pointer);
    curl_setopt($ch, CURLOPT_TIMEOUT, 120); // বড় ফাইলের জন্য ১২০ সেকেন্ড পর্যন্ত সময় দেওয়া হলো
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true); // রিডাইরেক্ট অনুসরণ করার জন্য
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true); // SSL সার্টিফিকেট যাচাই করা (নিরাপত্তার জন্য গুরুত্বপূর্ণ)
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);

    $result = curl_exec($ch);
    $error_no = curl_errno($ch);
    curl_close($ch);
    fclose($file_pointer);

    if ($result === false) {
        // cURL এরর হলে একটি সুনির্দিষ্ট বার্তা দিয়ে Exception throw করা হচ্ছে
        throw new Exception("cURL download failed. Error Code: " . $error_no . " - " . curl_strerror($error_no));
    }

    return true;
}

/**
 * একটি ফোল্ডার এবং তার ভেতরের সমস্ত ফাইল ও সাব-ফোল্ডার ডিলিট করার জন্য Helper Function.
 * @param string $dir যে ফোল্ডারটি ডিলিট করা হবে।
 * @return bool সফল হলে true।
 */
function recursive_delete($dir) {
    if (!file_exists($dir)) return true;
    if (!is_dir($dir)) return unlink($dir);
    foreach (scandir($dir) as $item) {
        if ($item == '.' || $item == '..') continue;
        if (!recursive_delete($dir . DIRECTORY_SEPARATOR . $item)) return false;
    }
    return rmdir($dir);
}

/**
 * সম্পূর্ণ আপডেট প্রক্রিয়াটি পরিচালনা করার মূল ফাংশন।
 * @return array আপডেটের ফলাফলসহ একটি associative array।
 */
function perform_update() {
    // PHP-কে বেশি সময় ধরে চলার এবং বেশি মেমরি ব্যবহারের অনুমতি দেওয়া হচ্ছে
    @set_time_limit(300);
    @ini_set('memory_limit', '256M');

    // প্রয়োজনীয় ফাইলগুলো লোড করা
    if (!defined('APP_VERSION')) require_once __DIR__ . '/config.php';
    global $pdo; // $pdo ভ্যারিয়েবলটিকে ফাংশনের ভেতরে অ্যাক্সেস করার জন্য
    if (!isset($pdo)) require_once __DIR__ . '/db_connect.php';

    // কনস্ট্যান্ট বা ধ্রুবক নির্ধারণ করা
    if (!defined('CENTRAL_API_URL')) define('CENTRAL_API_URL', 'https://settings.top/dashboard_project/api/check_update.php');
    if (!defined('UPDATE_LOG_FILE')) define('UPDATE_LOG_FILE', __DIR__ . '/update_log.txt');
    if (!defined('MAINTENANCE_FLAG')) define('MAINTENANCE_FLAG', __DIR__ . '/maintenance.flag');

    // লগ মেসেজ লেখার জন্য ফাংশন
    function log_message($message) {
        file_put_contents(UPDATE_LOG_FILE, date('[Y-m-d H:i:s] ') . $message . "\n", FILE_APPEND);
    }

    log_message("--- Starting Update Check (Current Version: " . APP_VERSION . ") ---");
    $project_root = __DIR__;

    try {
        // ধাপ ১: কেন্দ্রীয় সার্ভারে আপডেটের জন্য চেক করা
        $ch_check = curl_init();
        curl_setopt($ch_check, CURLOPT_URL, CENTRAL_API_URL);
        curl_setopt($ch_check, CURLOPT_POST, 1);
        curl_setopt($ch_check, CURLOPT_POSTFIELDS, http_build_query(['license_key' => LICENSE_KEY, 'current_version' => APP_VERSION]));
        curl_setopt($ch_check, CURLOPT_RETURNTRANSFER, true);
        $response = curl_exec($ch_check);
        curl_close($ch_check);
        $update_info = json_decode($response, true);

        if (!$update_info || !isset($update_info['update_available'])) {
            throw new Exception("Invalid response from update server: " . $response);
        }

        // ধাপ ২: যদি নতুন আপডেট পাওয়া যায়
        if ($update_info['update_available']) {
            log_message("Update found! New version: " . $update_info['version']);
            file_put_contents(MAINTENANCE_FLAG, 'Application is currently being updated.');
            log_message("Maintenance mode enabled.");
            
            // ধাপ ২.১: cURL ব্যবহার করে ম্যানিফেস্ট এবং জিপ ফাইল ডাউনলোড করা
            $manifest_path_temp = $project_root . '/manifest.json';
            $zip_file_path = $project_root . '/update_package.zip';

            log_message("Attempting to download manifest from: " . $update_info['manifest_url']);
            download_file_with_curl($update_info['manifest_url'], $manifest_path_temp);
            log_message("Manifest downloaded successfully.");

            log_message("Attempting to download ZIP package from: " . $update_info['download_url']);
            download_file_with_curl($update_info['download_url'], $zip_file_path);
            log_message("Update package downloaded successfully.");
            
            $manifest = json_decode(file_get_contents($manifest_path_temp), true);
            unlink($manifest_path_temp); // অস্থায়ী ম্যানিফেস্ট ফাইল ডিলিট করা

            // ধাপ ২.২: একটি অস্থায়ী ফোল্ডারে জিপ এক্সট্র্যাক্ট করা
            $temp_dir = $project_root . '/update_temp';
            if (is_dir($temp_dir)) {
                recursive_delete($temp_dir);
            }
            mkdir($temp_dir, 0777, true);

            $zip = new ZipArchive;
            if ($zip->open($zip_file_path) !== TRUE) throw new Exception("Failed to open ZIP file.");
            $zip->extractTo($temp_dir);
            $zip->close();
            unlink($zip_file_path);
            log_message("Package extracted to temporary directory.");

            // ধাপ ২.৩: ম্যানিফেস্ট অনুযায়ী ফাইল তুলনা এবং কপি করা
            $updated_files_count = 0;
            $opcache_files_to_invalidate = [];

            foreach ($manifest as $relative_path => $new_hash) {
                $source_file = $temp_dir . '/' . $relative_path;
                
                // জিপের ভেতরের 'product_package/' অংশটি বাদ দিয়ে পাথ তৈরি করা হচ্ছে
                $destination_relative_path = preg_replace('/^product_package\//', '', $relative_path);
                $destination_file = $project_root . '/' . $destination_relative_path;

                if (!file_exists($source_file)) {
                    log_message("WARNING: Source file '{$relative_path}' not found in ZIP. Skipping.");
                    continue;
                }

                $current_hash = file_exists($destination_file) ? md5_file($destination_file) : null;
                
                if ($current_hash !== $new_hash) {
                    
                    // ফাইল কপি করার আগে, তার প্যারেন্ট ডিরেক্টরি বা ফোল্ডারটি তৈরি করা হচ্ছে
                    $destination_dir = dirname($destination_file);
                    
                    // যদি ডিরেক্টরিটির অস্তিত্ব না থাকে...
                    if (!is_dir($destination_dir)) {
                        // ...তাহলে এটি তৈরি করো। `true` প্যারামিটারটি recursive ফোল্ডার তৈরি করতে সাহায্য করে
                        if (!mkdir($destination_dir, 0755, true)) {
                            log_message("CRITICAL ERROR: Could not create directory: " . $destination_dir);
                            continue; // এই ফাইলটি স্কিপ করে পরের ফাইলে যাও
                        }
                    }
                    
                    if (copy($source_file, $destination_file)) {
                        $updated_files_count++;
                        $opcache_files_to_invalidate[] = $destination_file;
                    } else {
                        log_message("WARNING: Could not copy updated file to: " . $destination_file);
                    }
                }
            }
            log_message("Differential update complete. Total files updated: " . $updated_files_count);
            
            // অস্থায়ী ফোল্ডার ডিলিট করা
            recursive_delete($temp_dir);
            log_message("Temporary directory deleted.");

            // ধাপ ২.৪: OPcache ক্লিয়ার করা
            if (function_exists('opcache_invalidate') && count($opcache_files_to_invalidate) > 0) {
                log_message("Invalidating OPcache for " . count($opcache_files_to_invalidate) . " files...");
                foreach ($opcache_files_to_invalidate as $file) {
                    opcache_invalidate($file, true);
                }
                log_message("OPcache invalidation complete.");
            }

            // ধাপ ২.৫: ডাটাবেস মাইগ্রেশন চালানো
            if (!empty($update_info['sql_migration'])) {
                log_message("Running database migration...");
                $pdo->exec($update_info['sql_migration']);
                log_message("Database migration completed.");
            }
            
            // ধাপ ২.৬: কনফিগ ফাইলে ভার্সন আপডেট করা
            $config_path = $project_root . '/config.php';
            $config_content = file_get_contents($config_path);
            $new_config_content = preg_replace("/define\s*\(\s*'APP_VERSION'\s*,\s*'.*?'\s*\);/", "define('APP_VERSION', '" . $update_info['version'] . "');", $config_content);
            file_put_contents($config_path, $new_config_content);
            log_message("Config file updated to version " . $update_info['version']);

            // রক্ষণাবেক্ষণ মোড বন্ধ করা
            if (file_exists(MAINTENANCE_FLAG)) unlink(MAINTENANCE_FLAG);
            log_message("Maintenance mode disabled. Update process finished successfully.");

            return ['success' => true, 'message' => 'Application updated successfully to version ' . $update_info['version'], 'new_version' => $update_info['version']];
        
        } else {
            log_message("No new updates found. Application is up-to-date.");
            return ['success' => true, 'message' => 'Your application is already up-to-date.', 'new_version' => APP_VERSION];
        }

    } catch (Exception $e) {
        log_message("!!! UPDATE FAILED: " . $e->getMessage());
        if (file_exists(MAINTENANCE_FLAG)) {
            unlink(MAINTENANCE_FLAG);
        }
        return ['success' => false, 'message' => $e->getMessage()];
    }
}