<?php
/**
 * API Endpoint for Initiating a Payment on a Self-Hosted Instance
 * Version 4.1 - Added Base URL validation for enhanced security.
 */

header('Content-Type: application/json');

// সেলফ-হোস্টেড সাইটের নিজস্ব ডাটাবেস সংযোগ স্থাপন করা
require '../db_connect.php';

try {
    // ধাপ ১: ইন্টিগ্রেটেড সাইট থেকে পাঠানো API কী (Authorization Header) গ্রহণ করা
    $headers = getallheaders();
    $auth_header = $headers['Authorization'] ?? '';
    if (!preg_match('/Bearer\s(\S+)/', $auth_header, $matches)) {
        http_response_code(401);
        throw new Exception('API key is missing or invalid in Authorization header.');
    }
    $api_key = $matches[1];

    // ★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★
    // ★★★               মূল সমাধান             ★★★
    // ★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★

    // ধাপ ২: API কী ব্যবহার করে user_id এবং ডাটাবেসে সেভ করা base_url খুঁজে বের করা
    $stmt = $pdo->prepare("SELECT user_id, base_url FROM settings WHERE api_key = ?");
    $stmt->execute([$api_key]);
    $setting = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$setting) {
        http_response_code(403); // Forbidden
        throw new Exception('Invalid API key.');
    }
    
    $user_id = $setting['user_id'];
    $stored_base_url = $setting['base_url'] ?? '';

    // ধাপ ৩: ইন্টিগ্রেটেড সাইট থেকে পাঠানো পেমেন্টের তথ্য (JSON Body) গ্রহণ করা
    $json_data = file_get_contents('php://input');
    $data = json_decode($json_data, true);

    $client_base_url = $data['client_base_url'] ?? '';

    // ধাপ ৪: ডাটাবেসে থাকা Base URL-এর সাথে ক্লায়েন্টের পাঠানো URL মেলানো
    if (empty($stored_base_url) || rtrim($stored_base_url, '/') !== rtrim($client_base_url, '/')) {
        
        // লগ ফাইলে এররটি লিখে রাখা হচ্ছে
        $log_file_path = __DIR__ . '/../error_log';
        $timestamp = date('d-M-Y H:i:s e');
        $log_entry = sprintf(
            "[%s] API Domain Mismatch: Expected URL='%s', Received URL='%s'\n",
            $timestamp,
            $stored_base_url,
            $client_base_url ?: '[URL Not Provided]'
        );
        file_put_contents($log_file_path, $log_entry, FILE_APPEND | LOCK_EX);

        // ক্লায়েন্টকে এরর মেসেজ পাঠানো হচ্ছে
        http_response_code(403);
        echo json_encode([
            'status' => 'error', 
            'message' => 'Domain mismatch. দুঃখিত, সংযোগ স্থাপন করা যায়নি।'
        ]);
        exit();
    }
    // ★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★
    // ★★★             সমাধান শেষ             ★★★
    // ★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★

    $amount = $data['amount'] ?? 0;
    if ($amount <= 0) {
        http_response_code(400);
        throw new Exception('Amount must be a positive number.');
    }

    // ধাপ ৫: নিজস্ব ডাটাবেসে নতুন ট্রানজেকশন তৈরি করা
    $unique_id = sha1(uniqid(rand(), true) . microtime()); 

    $sql = "INSERT INTO transactions (user_id, unique_id, amount, customer_name, customer_email, success_url, cancel_url, meta_data) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([
        $user_id,
        $unique_id,
        $amount,
        $data['customer_name'] ?? null,
        $data['customer_email'] ?? null,
        $data['success_url'] ?? null,
        $data['cancel_url'] ?? null,
        $data['meta_data'] ?? null
    ]);
    
    // ধাপ ৬: সেলফ-হোস্টেড সাইটের নিজস্ব চেকআউট URL তৈরি এবং ফেরত পাঠানো
    $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? "https" : "http";
    $host = $_SERVER['HTTP_HOST'];
    $base_path = rtrim(str_replace('/api/initiate_payment.php', '', $_SERVER['SCRIPT_NAME']), '/');

    $payment_url = "{$protocol}://{$host}{$base_path}/checkout.php?trx_id={$unique_id}";

    http_response_code(201); // 201 Created
    echo json_encode(['status' => 'success', 'payment_url' => $payment_url]);

} catch (Exception $e) {
    $statusCode = http_response_code();
    if ($statusCode < 400) { $statusCode = 500; }
    
    http_response_code($statusCode);
    echo json_encode([
        'status' => 'error', 
        'message' => $e->getMessage()
    ]);
}
?>