<?php
require 'auth_check.php';
require 'db_connect.php';

// ডাইনামিক URL এবং API Key আনা হচ্ছে
$user_id = $_SESSION['user_id'];
$stmt = $pdo->prepare("SELECT api_key, base_url FROM settings WHERE user_id = ?");
$stmt->execute([$user_id]);
$setting = $stmt->fetch(PDO::FETCH_ASSOC);

$api_key = $setting['api_key'] ?? 'YOUR_SECRET_API_KEY';
$base_url = $setting['base_url'] ?? 'https://your-domain.com/api'; // ফলব্যাক URL

$endpoint_initiate = "{$base_url}/initiate_payment.php";
$endpoint_verify = "{$base_url}/verify_payment.php";

include 'templates/header.php';
include 'templates/sidebar.php';
?>
<div class="page-container">
    <div class="page-header">
        <h1 class="page-title">Developer API Documentation (v2.1)</h1>
    </div>

    <!-- Introduction -->
    <div class="content-card">
        <h3>Introduction & Authentication</h3>
        <p>This documentation provides a comprehensive guide to integrating our payment gateway. All API requests must be authenticated by including your API key in the request header.</p>
        <pre><code>Authorization: Bearer <?php echo htmlspecialchars($api_key); ?></code></pre>
    </div>

    <!-- Payment Flow -->
    <div class="content-card">
        <h3>Payment Flow Overview</h3>
        <p>The integration process relies on the user's browser to confirm the payment. The flow follows these steps:</p>
        <ol>
            <li><strong>Step 1: Initiate Payment.</strong> Your server sends a secure request to our <code>/initiate_payment.php</code> endpoint, including your site's unique Base URL for verification.</li>
            <li><strong>Step 2: Redirect User.</strong> Our API validates your Base URL and responds with a unique <code>payment_url</code>. Your application must immediately redirect the user's browser to this URL.</li>
            <li><strong>Step 3: Customer Pays.</strong> The customer completes the payment on our secure checkout page.</li>
            <li><strong>Step 4: Return to Your Site.</strong> After a successful payment, our gateway redirects the user's browser back to the <code>success_url</code> you provided, appending the unique <strong><code>transaction_id</code></strong> as proof of payment.</li>
            <li><strong>Step 5: Verify on Your Server.</strong> Your script at the <code>success_url</code> reads this <code>transaction_id</code> and updates your database to mark the order as "Paid".</li>
        </ol>
    </div>

    <!-- 1. Initiate Payment -->
    <div class="content-card">
        <h3>1. Initiate a Payment</h3>
        <p>To start a payment, send a POST request to the Payment Initiation Endpoint. For security, you must include your site's Base URL to verify the request's origin.</p>
        <h4>Endpoint</h4>
        <pre><code>POST <?php echo htmlspecialchars($endpoint_initiate); ?></code></pre>

        <h4>Request Body (JSON)</h4>
        <pre><code>{
    "amount": 150.50,
    "client_base_url": "<?php echo htmlspecialchars($base_url); ?>",
    "customer_name": "John Doe",
    "customer_email": "john.doe@example.com",
    "success_url": "https://yourwebsite.com/order/success.php",
    "cancel_url": "https://yourwebsite.com/cart.php",
    "meta_data": "{\"order_id\": \"YOUR_UNIQUE_ORDER_ID_123\"}"
}</code></pre>
        <h4>Parameters</h4>
        <ul>
            <li><code>amount</code> (float, required): The total amount to be paid.</li>
            <li><code>client_base_url</code> (string, required): Your site's API Base URL. <strong>This must exactly match the "Base URL" found on your API Settings page.</strong></li>
            <li><code>customer_name</code> (string, optional): The customer's full name.</li>
            <li><code>customer_email</code> (string, optional): The customer's email address.</li>
            <li><code>success_url</code> (string, required): The URL on your site where the customer will be redirected after a successful payment.</li>
            <li><code>cancel_url</code> (string, required): The URL where the customer is sent if they cancel.</li>
            <li><code>meta_data</code> (JSON string, optional): A JSON string for any extra data, like your internal <code>order_id</code>.</li>
        </ul>
        <h4>Success Response (JSON)</h4>
        <pre><code>{
    "status": "success",
    "payment_url": "<?php echo str_replace('/api', '', htmlspecialchars($base_url)); ?>/checkout.php?trx_id=a1b2c3..."
}
// Your server must now redirect the user's browser to the value of "payment_url".</code></pre>
        <h4>Error Response (Example)</h4>
        <p>If the <code>client_base_url</code> does not match the one stored in your settings, the request will be rejected:</p>
        <pre><code>{
    "status": "error",
    "message": "Domain mismatch. দুঃখিত, সংযোগ স্থাপন করা যায়নি।"
}</code></pre>
    </div>

    <!-- 2. Handle the Return -->
    <div class="content-card">
        <h3>2. Handle the Customer's Return</h3>
        <p>After a successful payment, the user will be redirected to your <code>success_url</code> with the payment's unique transaction ID appended as a query parameter.</p>

        <h4>Example Return URL:</h4>
        <pre><code>https://yourwebsite.com/order/success.php?transaction_id=CJH1D2Y4W1</code></pre>
        
        <h4>What Your `success.php` Script Must Do:</h4>
        <ol>
            <li>Read the <code>transaction_id</code> from the URL's query parameters.</li>
            <li>Update your order's status to "Paid" or "Completed" in your database, storing the received <code>transaction_id</code> as proof of payment.</li>
            <li>Display a "Thank You" or order confirmation message to the customer.</li>
        </ol>
        
        <h4>Code Example (Plain PHP):</h4>
        <pre><code>&lt;?php
$transactionId = $_GET['transaction_id'] ?? null;

if ($transactionId) {
    //
    // 1. Find the order in your database that was pending.
    // 2. Update its status to 'completed'.
    // 3. Store the $transactionId as the payment reference.
    // 4. Show a success message to the customer.
    //
    echo "Thank you! Your payment with Transaction ID: " . htmlspecialchars($transactionId) . " was successful.";
} else {
    echo "Payment verification failed.";
}
?&gt;</code></pre>
    </div>

</div>

</div> <!-- .main-content -->
<script src="js/main.js?v=<?php echo filemtime('js/main.js'); ?>"></script>
</body>
</html>