<?php
/**
 * Unified License API Server
 * Handles requests from WHMCS (Provisioning) AND WordPress Plugins (Verification)
 */

header('Content-Type: application/json');
require_once 'config.php';
require_once __DIR__ . '/admin/includes/db.php';

// ------------------------------------------------------------------
// 1. SECURITY & CONFIGURATION CHECK
// ------------------------------------------------------------------

// URL থেকে Endpoint Key চেক করা (যেমন: /api.php?endpoint_key=YOUR_SECRET)
// অথবা POST ডাটা থেকে চেক করা
$sent_endpoint_key = $_REQUEST['endpoint_key'] ?? null;

// যদি config.php তে Key সেট করা থাকে, তবে অবশ্যই মিলতে হবে
if (defined('API_ENDPOINT_KEY') && !empty(API_ENDPOINT_KEY)) {
    if ($sent_endpoint_key !== API_ENDPOINT_KEY) {
        http_response_code(403);
        echo json_encode([
            'success' => false, 
            'status' => 'error', 
            'message' => 'Invalid API Endpoint Key. Access Denied.'
        ]);
        exit;
    }
}

// ডাটাবেস কানেকশন
try {
    $pdo = get_db_connection();
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Database Connection Failed']);
    exit;
}

// ------------------------------------------------------------------
// 2. HELPER FUNCTIONS
// ------------------------------------------------------------------

function sanitize_text($str) {
    return trim(strip_tags((string)$str));
}

function generate_unique_license_key($pdo) {
    $max_tries = 10;
    for ($i = 0; $i < $max_tries; $i++) {
        // ফরমেট: XXXXX-XXXXX-XXXXX-XXXXX
        $key = strtoupper(
            substr(md5(uniqid(rand(), true)), 0, 5) . '-' . 
            substr(md5(uniqid(rand(), true)), 0, 5) . '-' . 
            substr(md5(uniqid(rand(), true)), 0, 5) . '-' . 
            substr(md5(uniqid(rand(), true)), 0, 5)
        );
        
        $stmt = $pdo->prepare("SELECT id FROM licenses WHERE license_key = ?");
        $stmt->execute([$key]);
        if (!$stmt->fetch()) {
            return $key;
        }
    }
    return false;
}

function build_response($license, $activations, $status, $site_status = null) {
    $limit = (int)$license['activation_limit'];
    $data = [
        'item_name'         => $license['item_name'],
        'expires_at'        => !empty($license['expires_at']) ? date('Y-m-d H:i:s', strtotime($license['expires_at'])) : 'lifetime',
        'activation_limit'  => ($limit === 0) ? 'unlimited' : $limit,
        'activation_count'  => count($activations),
        'license_status'    => $status,
        'plan'              => $license['plan'] ?? null
    ];
    if ($site_status !== null) {
        $data['site_status'] = $site_status;
    }
    return $data;
}

// ------------------------------------------------------------------
// 3. MAIN LOGIC
// ------------------------------------------------------------------

$action = $_POST['action'] ?? null;
$response = ['success' => false, 'message' => 'Invalid Request'];

try {
    
    // ==============================================================
    // SECTION A: WHMCS MANAGEMENT ACTIONS (Provisioning)
    // ==============================================================

    if ($action === 'create_license') {
        // 1. নতুন লাইসেন্স তৈরি
        $item_name = sanitize_text($_POST['item_name'] ?? '');
        $plan = sanitize_text($_POST['plan'] ?? '');
        $limit = intval($_POST['activation_limit'] ?? 1);
        $expires_at = !empty($_POST['expires_at']) ? date('Y-m-d H:i:s', strtotime($_POST['expires_at'])) : null;
        
        // প্রোডাক্ট আইডি বের করা বা তৈরি করা
        $stmt = $pdo->prepare("SELECT id FROM products WHERE item_name = ?");
        $stmt->execute([$item_name]);
        $product = $stmt->fetch();
        
        if (!$product) {
            $stmt = $pdo->prepare("INSERT INTO products (item_name) VALUES (?)");
            $stmt->execute([$item_name]);
            $product_id = $pdo->lastInsertId();
        } else {
            $product_id = $product['id'];
        }

        // লাইসেন্স কি জেনারেট
        $license_key = generate_unique_license_key($pdo);
        
        if ($license_key) {
            $stmt = $pdo->prepare("INSERT INTO licenses (license_key, product_id, status, activation_limit, expires_at) VALUES (?, ?, 'active', ?, ?)");
            $stmt->execute([$license_key, $product_id, $limit, $expires_at]);
            
            // প্ল্যান আপডেট (টেবিলে কলাম থাকলে)
            // নোট: আপনার licenses টেবিলে যদি 'plan' কলাম থাকে তবে নিচের লাইনটি কাজ করবে
            // ALTER TABLE licenses ADD COLUMN plan VARCHAR(50) NULL;
            // $pdo->prepare("UPDATE licenses SET plan = ? WHERE license_key = ?")->execute([$plan, $license_key]);

            $response = [
                'success' => true,
                'license_key' => $license_key,
                'message' => 'License created successfully'
            ];
        } else {
            $response['message'] = 'Failed to generate unique license key';
        }

    } elseif ($action === 'manage_license') {
        // 2. লাইসেন্স স্ট্যাটাস বা মেয়াদ আপডেট
        $license_key = sanitize_text($_POST['license_key'] ?? '');
        $new_status = sanitize_text($_POST['status_action'] ?? ''); // active, suspended, terminated
        $new_expiry = !empty($_POST['expires_at']) ? date('Y-m-d H:i:s', strtotime($_POST['expires_at'])) : null;

        if ($license_key) {
            $query = "UPDATE licenses SET updated_at = NOW()";
            $params = [];

            if ($new_status) {
                $query .= ", status = ?";
                $params[] = $new_status;
            }
            // শুধু যদি এক্সপায়ারি ডেট পাঠানো হয় তবেই আপডেট হবে
            if (isset($_POST['expires_at'])) {
                $query .= ", expires_at = ?";
                $params[] = $new_expiry;
            }

            $query .= " WHERE license_key = ?";
            $params[] = $license_key;

            $pdo->prepare($query)->execute($params);
            $response = ['success' => true, 'message' => 'License updated successfully'];
        }

    } elseif ($action === 'get_license_details') {
        // 3. WHMCS ক্লায়েন্ট এরিয়ার জন্য তথ্য আনা
        $license_key = sanitize_text($_POST['license_key'] ?? '');
        
        $stmt = $pdo->prepare("SELECT l.*, p.item_name FROM licenses l JOIN products p ON l.product_id = p.id WHERE l.license_key = ?");
        $stmt->execute([$license_key]);
        $license = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($license) {
            $activations = json_decode($license['activations'], true) ?: [];
            $response = [
                'success' => true,
                'data' => [
                    'license_key' => $license['license_key'],
                    'status' => $license['status'],
                    'item_name' => $license['item_name'],
                    'activations' => $activations,
                    'activation_count' => count($activations),
                    'activation_limit' => ($license['activation_limit'] == 0) ? 'Unlimited' : $license['activation_limit'],
                    'expires_at' => $license['expires_at']
                ]
            ];
        } else {
            $response['message'] = 'License not found';
        }

    } elseif ($action === 'update_activations') {
        // 4. ডোমেইন রিমুভ করার জন্য (Client Area থেকে)
        $license_key = sanitize_text($_POST['license_key'] ?? '');
        $activations_json = $_POST['activations'] ?? '[]';

        $pdo->prepare("UPDATE licenses SET activations = ? WHERE license_key = ?")->execute([$activations_json, $license_key]);
        $response = ['success' => true, 'message' => 'Activations list updated'];
    }

    // ==============================================================
    // SECTION B: PLUGIN ACTIONS (Verification)
    // ==============================================================
    
    elseif (in_array($action, ['activate', 'deactivate', 'check'])) {
        
        $license_key = sanitize_text($_POST['license_key'] ?? '');
        $item_name   = sanitize_text($_POST['item_name'] ?? '');
        $site_url    = $_POST['url'] ?? '';

        // বেসিক ভ্যালিডেশন
        if (empty($license_key) || empty($item_name)) {
            echo json_encode(['success' => false, 'message' => 'Missing license key or item name', 'status' => 'error']);
            exit;
        }

        // ইউআরএল নরমালাইজেশন
        $domain = null;
        if ($site_url) {
            $parsed = parse_url('//' . preg_replace('#^https?://#', '', $site_url));
            $domain = isset($parsed['host']) ? strtolower($parsed['host']) : null;
            if ($domain && strpos($domain, 'www.') === 0) $domain = substr($domain, 4);
        }

        // লাইসেন্স খোঁজা
        $stmt = $pdo->prepare("SELECT l.*, p.item_name FROM licenses l JOIN products p ON l.product_id = p.id WHERE l.license_key = ?");
        $stmt->execute([$license_key]);
        $license = $stmt->fetch(PDO::FETCH_ASSOC);

        // ১. লাইসেন্স ভ্যালিড কিনা
        if (!$license) {
            echo json_encode(['success' => false, 'message' => 'Invalid license key', 'status' => 'invalid']);
            exit;
        }

        // ২. আইটেম নেম মিলছে কিনা
        if (strcasecmp($license['item_name'], $item_name) !== 0) {
            echo json_encode(['success' => false, 'message' => 'Item name mismatch', 'status' => 'invalid_item']);
            exit;
        }

        // ৩. মেয়াদ শেষ কিনা
        if (!empty($license['expires_at']) && strtotime($license['expires_at']) < time()) {
            // স্ট্যাটাস Expired করে দেওয়া
            if ($license['status'] === 'active') {
                $pdo->prepare("UPDATE licenses SET status = 'expired' WHERE id = ?")->execute([$license['id']]);
                $license['status'] = 'expired';
            }
            $activations = json_decode($license['activations'], true) ?: [];
            echo json_encode(['success' => false, 'message' => 'License expired', 'status' => 'expired', 'data' => build_response($license, $activations, 'expired')]);
            exit;
        }

        // ৪. স্ট্যাটাস চেক (Suspended/Terminated)
        if ($license['status'] !== 'active' && $action !== 'deactivate') {
            $activations = json_decode($license['activations'], true) ?: [];
            echo json_encode(['success' => false, 'message' => 'License is ' . $license['status'], 'status' => $license['status'], 'data' => build_response($license, $activations, $license['status'])]);
            exit;
        }

        // অ্যাকটিভেশন ডাটা প্রসেসিং
        $activations = json_decode($license['activations'], true) ?: [];
        $current_count = count($activations);
        $limit = (int)$license['activation_limit'];

        // --- PLUGIN ACTION: ACTIVATE ---
        if ($action === 'activate') {
            if (!$domain) {
                echo json_encode(['success' => false, 'message' => 'Invalid site URL']); exit;
            }

            if (isset($activations[$domain])) {
                $response = ['success' => true, 'message' => 'Already active', 'status' => 'active', 'data' => build_response($license, $activations, 'active', 'active')];
            } elseif ($limit > 0 && $current_count >= $limit) {
                $response = ['success' => false, 'message' => 'Activation limit reached', 'status' => 'limit_reached', 'data' => build_response($license, $activations, 'active', 'limit_reached')];
            } else {
                // নতুন ডোমেইন যোগ করা
                $activations[$domain] = [
                    'status' => 'activated',
                    'activation_date' => date('Y-m-d H:i:s')
                ];
                $pdo->prepare("UPDATE licenses SET activations = ? WHERE id = ?")->execute([json_encode($activations), $license['id']]);
                
                $response = ['success' => true, 'message' => 'Activated successfully', 'status' => 'active', 'data' => build_response($license, $activations, 'active', 'active')];
            }
        }

        // --- PLUGIN ACTION: DEACTIVATE ---
        elseif ($action === 'deactivate') {
            if ($domain && isset($activations[$domain])) {
                unset($activations[$domain]);
                $pdo->prepare("UPDATE licenses SET activations = ? WHERE id = ?")->execute([json_encode($activations), $license['id']]);
            }
            $response = ['success' => true, 'message' => 'Deactivated successfully', 'status' => 'deactivated', 'data' => build_response($license, $activations, $license['status'], 'inactive')];
        }

        // --- PLUGIN ACTION: CHECK ---
        elseif ($action === 'check') {
            $site_status = ($domain && isset($activations[$domain])) ? 'active' : 'inactive';
            $response = ['success' => true, 'message' => 'License is valid', 'status' => 'active', 'data' => build_response($license, $activations, 'active', $site_status)];
        }

    } else {
        $response['message'] = 'Unknown Action';
    }

} catch (Exception $e) {
    $response['message'] = 'Server Error: ' . $e->getMessage();
}

echo json_encode($response);
exit;
?>